

### Project: IADB Government Payroll Analytics - Country
### Project leader: Dr Christian Schuster
### Code author (s): Robert Lipińskialcal
### Date last update: (run below)
file.info(rstudioapi::getActiveDocumentContext()$path)$mtime

### Script purpose: conducts cleaning of position names (collapsing person-contract observations to speed up implementation)

### Execution time: ~10 minutes

### Inputs: 
# 1) /data/intermediate/country_04_limpiar_cols.[format1]


### Outputs:
# 1) /data/intermediate/country_05_limpiar_puesto.[format1]
# *) /data/raw_qs/intermediate_temp/country_05_limpiar_puesto (temp).[format1] (only temporary file to avoid re-running full script in case of an error
# not necessary for executing the script)



#
# SET-UP --------------------------------------------------------------------------------------------
#

### source global file with packages, functions and global parameters
source(file.path(dirname(rstudioapi::getActiveDocumentContext()$path), '00_country_global.R'))


### create copy of the script
file.copy(rstudioapi::getSourceEditorContext()$path,
          gsub('code', 'code/00_archive', gsub('\\.R', ' - copy.R', rstudioapi::getSourceEditorContext()$path)),
          overwrite = T, copy.date = T)





## ' --------------------------------------------------------------------------------------------------------------------------------
# CLEAN POSITIONS -----------------------------------------------------------------------------------------------------------------
# 

t0 = Sys.time() # record start time

### read the file

## compare names in the data file to the ones used in the script
col_select1 = c('row_id_org', 'tipo_estamento_comprimido', 'tipo_calificacion', 'tipo_cargo',
                'organismo_nombre_clean', 'organismo_sector_comprimido')

col_names = names(open_dataset(file.path(main_dir, 'data', 'intermediate', "country_04_limpiar_cols.parquet")))
col_names %>% sort

country = read_flex(file.path(main_dir, 'data', 'intermediate', "country_04_limpiar_cols"), format = 'parquet', 
                  col_select = col_select1[col_select1 %in% col_names])

add1 = read_flex(file.path(main_dir, 'data', 'intermediate', "country_03_limpiar_conjunto (out)"), format = 'parquet', 
                 col_select = c('row_id_org', setdiff(col_select1, col_names)))


country = add1[country, on = 'row_id_org']
rm(add1)

## set as DT if not already done
if(!any(grepl('data.table', class(country)))){setDT(country)}
gc()


### 1) replace with tipo_calificacion where warranted --------------------------------------------------------------------------------------------------------------------
### around 12% of tipo_cargo entries are either NA or 'cargo no definido' - in those instances replace with tipo_califiacion, wheere
### it actually describes cargo not education level -> should take it down to ~0.8% (prior to removing distinct())

clavos_edu = paste(c('\\bensenanza', 'educacion', 'licencia',
                    '\\bmedio', '\\bmedia', '\\btitulo',
                   'estudios', 'completa'), collapse='|')

pr_top(country[is.na(tipo_cargo) & !str_detect(tipo_calificacion, clavos_edu),],
       'tipo_calificacion', 100)


country[, tipo_cargo := fifelse(str_detect(tipo_cargo, '^0$|cargo no definido'), NA, tipo_cargo)]
country[, tipo_cargo := fifelse(test = (is.na(tipo_cargo) ) &
                                            !str_detect(tipo_calificacion, clavos_edu), 
                                  tipo_calificacion, tipo_cargo)]

pr_isna(country$tipo_cargo)
# pr_isna(country$tipo_cargo2)


### check particular sub-string in tipo_calificacion
pr_top(country[str_detect(tipo_calificacion, '\\bmedi'),],
       'tipo_calificacion', 100)


### check particular sub-string in tipo_cargo
pr_top(country[str_detect(tipo_cargo, '\\bcomplet'),],
       'tipo_cargo', 20)

### check most common position names
pr_top(country, 'tipo_cargo', 100)

# table(is.na(country$tipo_cargo) ,
#       str_detect(country$tipo_calificacion, '\\bensenanza|educacion|licencia'), useNA='ifany')


# country[, tipo_cargo:=tipo_cargo2]
# country[, tipo_cargo2:=NULL]




### 2) nest  --------------------------------------------------------------------------------------------------------------
print('nest')

country[, n := fdistinct(row_id_org), by = tipo_cargo]
dim(country)

cols_no_rowid <- setdiff(names(country), "row_id_org")

country_puesto <- country[, .(id_nest = list(row_id_org)), by = cols_no_rowid]



### 3) remove common stopwords  ----------------------------------------------------------------------------------------
print('stopwords')

stopwords_spanish <- unique(c("de", 'del', "la", "los", "las", "les", "ex", "al", "el", "en", "y", "un", "una", "es",
                       'segun', 'sobre', 'vice', 'con', "y", "o", "la", "las", "el", "los", "un", "una", "unos", "unas",
                       "de", "del", "al", "en", "con", "por", "para", "a", "e", "u", "ni", "que", "se"))

pattern <- paste0("\\b(", paste(stopwords_spanish, collapse = "|"), ")\\b")

country_puesto$tipo_cargo2 <- sf_gsub(country_puesto$tipo_cargo, pattern, "")

# how does it compare to str_replace_all_regex(vectorize = F)

### remove all non-letter characters
country_puesto$tipo_cargo2 <- sf_gsub(country_puesto$tipo_cargo2, '[^a-z]', " ")

### remove single letters
country_puesto$tipo_cargo2<- sf_gsub(country_puesto$tipo_cargo2, "\\b[a-zA-Z]\\b", "")

### remove whitespaces
country_puesto$tipo_cargo2 <- sf_gsub(country_puesto$tipo_cargo2, '\u00A0', ' ', fixed=T)
country_puesto$tipo_cargo2 <- str_squish(country_puesto$tipo_cargo2)
country_puesto$tipo_cargo2 <- str_trim(country_puesto$tipo_cargo2)


### 4) leave only first n words (or all words if there are <n in total)   ----------------------------------------------------------------------------------------
### there cleaning function applied later doesn't deal well with proper names 
print('first n words')

n = 3

country_puesto$tipo_cargo3 <- sapply(strsplit(country_puesto$tipo_cargo2, " "), function(x) {
  paste(x[1:min(n, length(x))], collapse = " ")
})

country_puesto$tipo_cargo3 =   gsub("\\s+", " ", trimws(str_trim(country_puesto$tipo_cargo3)))

### (?) if raw cargo = 'NA' then NA? --------------------------------------------------------------------------------------------------------------------------------------------------------
## NOTE: basically ensures that tipo_calificacion NOT used instead (can always do it laters)
# country_puesto$tipo_cargo3 = ifelse(is.na(country_puesto$tipo_cargo), NA, country_puesto$tipo_cargo3)


### 5) clean common shortcuts and spare words --------------------------------------------------------------------------------------------------------------------
country_puesto$tipo_cargo4 = country_puesto$tipo_cargo3
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4, paste0('\\b', 'enc', '\\b'), 'encargado')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4, paste0('\\b', 'tec', '\\b'), 'tecnico')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4, paste0('\\b', 'tecn', '\\b'), 'tecnico')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4, paste0('\\b', 'jefo', '\\b'), 'jefe')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4, paste0('\\b', 'ing', '\\b'), 'ingeniero')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4, paste0('\\b', 'cs', '\\b'), 'ciencias')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4, paste0('\\b', 'ax', '\\b'), 'auxiliar')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4, paste0('\\b', 'aux', '\\b'), 'auxiliar')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4, paste0('\\b', 'ayud', '\\b'), 'ayudante')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4, paste0('\\b', 'partamento', '\\b'), 'departamento')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4, paste0('', 'dpto', ''), 'departamento')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4, paste0('\\b', 'depto', '\\b'), 'departamento')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4, paste0('\\b', 'administrativamente', '\\b'), 'administrativo')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4, paste0('\\b', 'div', '\\b'), 'division')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4, paste0('\\b', 'serv', '\\b'), 'servicios')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4, paste0('\\b', 'secc', '\\b'), 'seccion')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4, paste0('\\b', 'sec', '\\b'), 'secretario')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4, paste0('\\b', 'enfermeroo', '\\b'), 'enfermero')

country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4, paste0('\\b', 'adm', '\\b'), 'administrativo')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4, paste0('\\b', 'medicino', '\\b'), 'medico')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4, paste0('\\b', 'cirugio', '\\b'), 'cirujano')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4, paste0('\\b', 'cirujano', '\\b'), 'cirujano')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4, paste0('\\b', 'enf', '\\b'), 'enfermero')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4, paste0('\\b', 'tens', '\\b'), 'tecnico enfermeria nivel superior')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4, paste0('\\b', 'tns', '\\b'), 'tecnico nivel superior')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4, paste0('\\b', 'ed', '\\b'), 'educacion')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4, paste0('\\b', 'edu', '\\b'), 'educacion')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4, paste0('\\b', 'educ', '\\b'), 'educacion')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4, paste0('\\b', 'mat', '\\b'), 'matematica')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4, paste0('\\b', 'esc', '\\b'), 'escolar')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4, paste0('\\b', 'gral', '\\b'), 'general')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4, paste0('\\b', 'pie', '\\b'), 'programa integracion escolar')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4, paste0('\\b', 'coord', '\\b'), 'coordinador')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4, paste0('\\b', 'doc', '\\b'), 'docente')
# country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4, paste0('\\b', 'mant', '\\b'), 'mantenimiento')
# country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4, paste0('\\b', 'pas', '\\b'), 'pasante') # for medicos = trainee/internt


# country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4, paste0('\\b', 'dir', '\\b'), 'directivo') # run below to check that almost all 'dir' belong to
# 'directivo' rank -> no, 'dir' needs to also stand for something else, like  dirección/directorado?
sf(country_puesto$tipo_estamento_comprimido[grepl('\\bensenanza\\b', country_puesto$tipo_cargo4)])


### checks anything else before re-coding above?
sf(country_puesto$tipo_cargo4[grepl('\\bensenanza\\b', country_puesto$tipo_cargo4)])


### replace words not needed
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4,paste0('\\b', 'subdepartamento', '\\b'), '')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4,paste0('\\b', 'departamento', '\\b'), '')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4,paste0('\\b', 'unidad', '\\b'), '')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4,paste0('\\b', 'subdireccion', '\\b'), '')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4,paste0('\\b', 'servicio', '\\b'), '')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4,paste0('\\b', 'centro', '\\b'), '')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4,paste0('\\b', 'general', '\\b'), '')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4,paste0('\\b', 'sud', '\\b'), '')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4,paste0('\\b', 'soporte', '\\b'), '')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4,paste0('\\b', 'se', '\\b'), '')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4,paste0('\\b', 'no', '\\b'), '')
country_puesto$tipo_cargo4 = sf_gsub(country_puesto$tipo_cargo4,paste0('\\b', 'paro', '\\b'), '')

pr_isna(country_puesto$tipo_cargo4)



### 6) 'de-genderize' job titles -------------------------------------------------------------------------------
print('de-genderize')

standardize_job_title <- function(title) {
  # Replace common feminine endings with masculine ones
  title <- sf_gsub(title, "ora\\b", "or")  # e.g., "doctora" -> "doctor"
  title <- sf_gsub(title, "era\\b", "ero")  # e.g., "jardinera" -> "jardinero"
  title <- sf_gsub(title, "ina\\b", "ino")  # e.g., "medicina" -> "medicino"
  title <- sf_gsub(title, "a\\b|a$", "o")    # all -a endings
  return(title)
}

country_puesto$tipo_cargo5 = standardize_job_title(country_puesto$tipo_cargo4)

### re-apply short-cut cleaning as 'cleantext()' above reverses some of them
country_puesto$tipo_cargo5 = sf_gsub(country_puesto$tipo_cargo5, paste0('\\b', 'jefo', '\\b'), 'jefe')
country_puesto$tipo_cargo5 = sf_gsub(country_puesto$tipo_cargo5, paste0('\\b', 'medicino', '\\b'), 'medico')
country_puesto$tipo_cargo5 = sf_gsub(country_puesto$tipo_cargo5, "\\bgarz\\w*", "garzon")
country_puesto$tipo_cargo5 = sf_gsub(country_puesto$tipo_cargo5, "(a)", "", fixed=T)


country_puesto$tipo_cargo5 =   gsub("\\s+", " ", trimws(str_trim(country_puesto$tipo_cargo5)))


### (*)checks - how standardized the names become
fdistinct(country_puesto$tipo_cargo)  # from ~522k unique names
fdistinct(country_puesto$tipo_cargo5) # to ~189k unique names

### 7) professor vs professional-------------------------------------------------------------
print('professor')

# if name has any of those -> then it's professor, not professional 
# words not ending 'properly' due to spelling errors in the data
prof_edu = paste('\\beduc', '\\bedu\\b',
                 'matemat', '\\bmat\\b',
                 'programa integracion escolar',
                 'biolog', 'fisic', 'ingles', 
                 'quimica', 'artes', 'historia', 'frances', 'filosof', 
                 'cienci', 'religi', 'literatu', 'lenguaj', 'castell',
                 'music', 'cs  sociales', 'pedagog',
                 'catolica', 'evangelica',
                 sep = '|')


# profesor or profession -> assign professor if...
# note: cleantext() shouldn't change 'prof' into anything else
country_puesto[, tipo_cargo5 := fifelse(
                test = str_detect(tipo_cargo5, prof_edu),  #  ...the name contains the following (defined above)
                yes  = sf_gsub(tipo_cargo5, paste0('\\b', 'prof', '\\b'), 'profesor'),
                no   = sf_gsub(tipo_cargo5, paste0('\\b', 'prof', '\\b'), 'profesional'))]


### final clean names
country_puesto$tipo_cargo_clean = clean_text(country_puesto$tipo_cargo5)
country_puesto = country_puesto %>% relocate(tipo_cargo_clean, .after = 'tipo_cargo')
country_puesto$tipo_cargo_clean =   gsub("\\s+", " ", trimws(str_trim(country_puesto$tipo_cargo_clean)))
country_puesto[, tipo_cargo_clean := fifelse(tipo_cargo_clean == 'na', NA, tipo_cargo_clean)] 

# country_puesto_save = country_puesto

### （*）checks
### most common cargo categories in the cleaned version
temp = pr_top(country_puesto, 'tipo_cargo_clean', 100)

### what % of obs comprised by 100 most common positions?
temp = unique(country_puesto[tipo_cargo_clean %in% temp$tipo_cargo_clean, .(tipo_cargo_clean, n)])
fsum(temp$n)/nrow(country) # ~41.8%



### 8) rank clean  ------------------------------------------------------------------------------------------------------
# country_puesto = country_puesto_save
setDT(country_puesto)


#### + 'cargo no definido' ------------------------------------------------------------------------------------------------------------------
country_puesto[, tipo_estamento_comprimido := sf_gsub(tipo_estamento_comprimido, 'é', 'e')]
country_puesto[, tipo_estamento_comprimido := fifelse(is.na(tipo_estamento_comprimido), 'estamento no definido', tipo_estamento_comprimido)]




### > save (temp) ------------------------------------------------------------------------------------------------------------------


# country_puesto = country_puesto %>% select(-any_of(c('tipo_cargo2', 'tipo_cargo3', 'tipo_cargo4', 'tipo_cargo5')))

## assign puesto ID to easier identify rows
# country_puesto$puesto_id = 1:nrow(country_puesto)
country_puesto[, puesto_id := .I]

# write_flex(country_puesto, file.path(main_dir, 'data', 'intermediate_temp', 'country_05_limpiar_puesto (temp)'), format='rds')
write_flex(country_puesto, file.path(main_dir, 'data', 'intermediate_temp', 'country_05_limpiar_puesto (temp)'), format=format1)


beep()




# ' --------------------------------------------------------------------------------------------------------------------------------
# DEFINE POSITIONS -----------------------------------------------------------------------------------------------------------------
#
print('define position')

# country_puesto = read_flex(file.path(main_dir, 'data', 'intermediate_temp', 'country_05_limpiar_puesto (temp)'), format='rds')
country_puesto = read_flex(file.path(main_dir, 'data', 'intermediate_temp', 'country_05_limpiar_puesto (temp)'), format=format1)



#### + docente ----------------------------------------------------------------------------------------------------

# maestro? -> common term for teachers in other LAC countries, like the Dominican Republic, but NOT in Country
# pr_top(country_puesto[str_detect(country_puesto$tipo_cargo_clean, 'maestro'),], 'tipo_cargo_clean', 60)

clavos_docente = str_trim(paste(c('\\bdocent', 'pedagog',
                                          # 'licencia', 'licenciado', 'magister', # 'educacion media',
                                          'professor', 'profesor', 'docent', 'educador',
                                          'tutor', 'parvulario'
                                          # 'ensenanzo medio', 'ensenanzo', 'ensenanza', ### NOT! It's education level of a civil servant
                                          ), # suggested by the government (adding \\b and/or keeping professor doesn't really matter)
                                        collapse = '|\\b'))

# but we also exclude strings that also contain
clavos_NOT_docente = str_trim(paste(c('paradocente'),
                                      collapse = '|\\b')) # but not nurses or paramedics or any of the miscellaneous words (mainly for 'cargo no definido' rank)



# classify the position as a dummy column
country_puesto = country_puesto %>% mutate(puesto_docente = fifelse(test = (str_detect(tipo_cargo_clean, clavos_docente)) &
                                                                        !str_detect(tipo_cargo_clean, clavos_NOT_docente) &
                                                                        !str_detect(organismo_nombre_clean, 'universidad|escuela superior|instituto|politecnica') &
                                                                          (!is.na(tipo_cargo_clean)),
                                                                      yes = 1,
                                                                      no = 0
          ))

# what names assigned to this class?
pr_top(country_puesto[country_puesto$puesto_docente == 1 & !is.na(country_puesto$puesto_docente)],
       'tipo_cargo_clean',20)

pr_top(country_puesto[country_puesto$puesto_docente == 1 & !is.na(country_puesto$puesto_docente) & str_detect(country_puesto$tipo_cargo_clean, 'directo')],
       'tipo_cargo_clean',20)


# *checks > how many IDs assigned to teachers? R: 19.7% of all observations are teachers - seems legitimate?
country_puesto %>% group_by(puesto_docente) %>%  summarise(n = n_distinct(unlist(id_nest))) %>% mutate(share = n/sum(n))


# note that the sum above is a bit tricky to interpret, because there are a lot of 
# teacher grados and so one person moving up those grados would be counted separately
# A more sensible approach is to subset full data to just the puesto names that are left
# as teacher ones and then count unique person_id, as done below



#### + // TI ----------------------------------------------------------------------------------------------------
# clavos_ti =  str_trim(paste(c('\\bprogrami', 'programador', 'codificad', 'informat', # is this not missing too many relevant roles?
#                               'software', 'reds', 'ciber', 'computacion',
#                               'datos', 'redes', 'red\\b',
#                               'ti\\b', 'it\\b'
#                               # ordernador - checked, doesn't occur in puesto names
#               ), # 
#               collapse = '|\\b'))
# 
# 
# # classify the position as a dummy column
# country_puesto = country_puesto %>% mutate(puesto_ti = fifelse(test = (str_detect(tipo_cargo_clean, clavos_ti))  &
#                                                                   (!is.na(tipo_cargo_clean)),
#                                                                 yes = 1,
#                                                                 no = 0
# ))
# 
# 
# # what names assigned to this class?
# pr_top(country_puesto[country_puesto$puesto_ti == 1 & !is.na(country_puesto$puesto_ti)],
#        'tipo_cargo_clean',20)

# *checks > how many IDs assigned to teachers? R: 19.7% of all observations are teachers - seems legitimate?
# country_puesto %>% group_by(puesto_docente) %>%  summarise(n = n_distinct(unlist(id_nest))) %>% mutate(share = n/sum(n))



#### + enfermero ----------------------------------------------------------------------------------------------------
clavos_enfermero =  str_trim(paste(c('enfermeria', 'enfermerio', 'enfermero', 'enfermera', 
                                     'atencion clin', 'atencion pacient',
                                     'matron'),
                                   collapse = '|\\b'))

# but we also exclude strings that also contain
clavos_NOT_enfermero = str_trim(paste(c('directivo', 'directivo', 'director', 
                                        'asesor', 'coordinator', 'encargado', 'supervisor',
                                        # 'tecnico', 
                                        'docente', 'secretar'),
                                      collapse = '|\\b')) # but not nurses or paramedics or any of the miscellaneous words (mainly for 'cargo no definido' rank)


# pr_top(country_puesto[str_detect(tipo_cargo_clean, 'atencion'),], 'tipo_cargo_clean', 30) # what are titles of 'atencion


# classify the position as a dummy column
country_puesto = country_puesto %>% mutate(puesto_enfermero = fifelse(test = str_detect(tipo_cargo_clean, clavos_enfermero) &
                                                                  !str_detect(tipo_cargo_clean, clavos_NOT_enfermero) &
                                                                  !str_detect(tipo_estamento_comprimido, 'directivo') &
                                                                  !is.na(tipo_cargo_clean),
                                                                yes = 1,
                                                                no = 0))



# 
# # what names assigned to this class?
pr_top(country_puesto[country_puesto$puesto_enfermero == 1 & !is.na(country_puesto$puesto_enfermero)],
       'tipo_cargo_clean', 20)


# *checks > how many IDs assigned to nurses? R: 4,9% of all observations are nurses - seems legitimate?
country_puesto %>% group_by(puesto_enfermero) %>%  summarise(n = n_distinct(unlist(id_nest))) %>% mutate(share = n/sum(n))



#### + medico ----------------------------------------------------------------------------------------------------
clavos_medico = str_trim(paste(c('\\bmedic[oa]', 'cirujan', 'cirugio', 'docto', 'anestesi'), collapse = '|\\b'))
### 'dr\\.?\\b', 'dra\\.?\\b' doesn't seem to yield correct names + dr is also abbrevation for 'district'


# add pre-defined strings (mainly specific medical specializations)
clavos_medico_add = read_flex(file.path('data', 'clean', 'dictionaries', 'country_puesto_add_strings'), format='xlsx') %>% 
  filter(puesto_clave == 'medico') %>% 
  filter(!strings %in% ('psicologo')) %>% 
  pull(strings) %>% paste0(., collapse = '|')

clavos_medico = paste0(clavos_medico, '|', clavos_medico_add)

#str_detect(c('medica general', 'daw', 'medico gene', 'medico general'), 'medic[oa] general|anestesiologo')

clavos_NOT_medico2 = paste(c('medicamentos', 'dispositivos', 'technolog', 'tecnolog', 
                                    'auxiliar', 'asistente', 'paramedic', 'enfermer', '\\bsecretar', 
                                    '\\brecept|recepcion|gestion|\\bcobro'), 
                                  collapse = '|') # but not nurses or paramedics or any of the miscellaneous words (mainly for 'cargo no definido' rank)

clavos_NOT_medico =  paste(c('medicamentos', 'dispositivos', 'technolog', 'tecnolog', 'tecnico',
                             'auxiliar', 'asistente', 'paramedic', 'enfermer', '\\bsecretar', 
                             'veterinar', 'odontolog', 'dentis', 'kinesiol', 'psicolog', # ensure those don't count (see Christian's email from 04/09/2025)
                             'directivo', 'directivo', 'director', 
                             'asesor', 'coordinator', 'encargado', 'supervisor',
                             'tecnico', 'docente', 'secretar',
                             '\\brecept|recepcion|gestion|\\bcobro'), 
                           collapse = '|') # but not nurses or paramedics or any of the miscellaneous words (mainly for 'cargo no definido' rank)


### classify the position as a dummy column
country_puesto = country_puesto %>%
  mutate(puesto_medico = fifelse(test = (str_detect(tipo_cargo_clean, clavos_medico)) &
                                      !(str_detect(tipo_cargo_clean, clavos_NOT_medico)) & 
                                      !(str_detect(tipo_cargo_clean, clavos_enfermero)) & # not any of the nurses strings
                                      !(tipo_estamento_comprimido %in% c('directivo', 'jefatura', # ... but not bosses
                                                                         'administrativo', # nor admin staff
                                                                         'auxiliar/asistente', # nor support staff
                                                                         'tecnico', # nor technicians xxx???
                                                                         'docente')) &
                                      !(is.na(tipo_cargo_clean)),
                                    yes = 1,
                                    no = 0
  ))


# what names assigned to this class?
pr_top(country_puesto[country_puesto$puesto_medico == 1 & !is.na(country_puesto$puesto_medico)],
       'tipo_cargo_clean', 20)


# *checks > how many IDs assigned to medicos? R: % of all observations are nurses - seems legitimate?
country_puesto %>% group_by(puesto_medico) %>%  summarise(n = n_distinct(unlist(id_nest))) %>% mutate(share = n/sum(n))



#### + secretario/auxiliar ----------------------------------------------------------------------------------------------------
clavos_secretario = str_trim(paste(c('^secretar',
                                     'apoyo administrativ[oa]', 'administrativ[oa] apoyo',
                                     'recepcioni',
                                     'asistente administrativ[oa]'
                                     # 'apoyo', 'auxiliar' # those not - see Christian's email from 04/09
                                     ),
                               collapse = '|'))
beep()

# pr_top(country[str_detect(tipo_cargo, 'ecretario administrativ')], 'tipo_estamento_comprimido', 30)



clavos_NOT_secretario = paste(c('superior', 
                                # 'ejecutiv', 'gestion',  'direccion',
                                'secretario juzgado policio', 
                                'auxiliar enfermerio', 'dentis', 'dental', # not auxiliar for nurses etc.!
                                # 'directivo', 'directivo', 'director',
                                'asesor', 'coordinator', 'encargado', 'supervisor',
                                'tecnico', 
                                # also not any teacher assistatns...
                                'docente', 'educacion', 'parvulo', 'escuelo', 'escolar', 'aulo', 'extension horar',
                                'salo', 'alumnos', 'liceo', 'colegio', 'di'
                                ),
                              collapse = '|')


### classify the position as a dummy column
country_puesto = country_puesto %>%
  mutate(puesto_secretario = fifelse(test = (str_detect(tipo_cargo_clean, clavos_secretario)) &
                                   !(str_detect(tipo_cargo_clean, clavos_NOT_secretario)) & 
                                   !(str_detect(tipo_cargo_clean, clavos_docente)) & 
                                   (tipo_estamento_comprimido %in% c( 'administrativo',
                                                                      'auxiliar/asistente',
                                                                      'estamento no definido')) &
                                   !(is.na(tipo_cargo_clean)),
                                 yes = 1,
                                 no = 0
  ))


# what names assigned to this class?
pr_top(country_puesto[country_puesto$puesto_secretario == 1 & !is.na(country_puesto$puesto_secretario)],
       'tipo_cargo_clean', 30)


# *checks > how many IDs assigned to secretarios? R: % of all observations are secretarios - seems legitimate?
country_puesto %>% group_by(puesto_secretario) %>%  summarise(n = n_distinct(unlist(id_nest))) %>% mutate(share = n/sum(n))



### > what's left? (in non-matched names)  ----------------------------------------------------------------------------------------------------------

country_puesto = country_puesto %>% 
  mutate(temp = rowSums(across(c(puesto_docente, puesto_medico, puesto_enfermero, puesto_secretario))))

temp = country_puesto %>% 
  filter(temp == 0) %>% 
  group_by(tipo_cargo_clean) %>% 
  summarise(id_count = n_distinct(unlist(id_nest))) %>% 
  arrange(desc(id_count))  %>% 
  mutate(share = 100*id_count/sum(id_count))

print(temp, n=60)


# pr_top(country_final[str_detect('otro tipo de cargo', country_final$puesto_clave), ], 'puesto_nombre', 30)
# pr_na(country_final$puesto_clave)
# beep(4)

# temp = country_puesto[tipo_cargo_clean == 'medicos',]
# pr_top(temp, 'tipo_calificacion', 30)

### >>> save what's left for Christian to browse -------------------------------------------------------------------------------------------------------------------------
write_flex(temp %>% slice(1:50),
           file.path('data', 'clean', 'aggregates', 'country non-classified positions.xlsx'))



### >>> save what's classified Christian to browse -------------------------------------------------------------------------------------------------------------------------
temp = rbindlist(list(
  pr_top(country_puesto[puesto_docente == 1], 'tipo_cargo_clean', 50) %>% mutate(puesto_clave = 'docente'),
  pr_top(country_puesto[puesto_medico == 1], 'tipo_cargo_clean', 50) %>% mutate(puesto_clave = 'medico'),
  pr_top(country_puesto[puesto_enfermero == 1], 'tipo_cargo_clean', 50) %>% mutate(puesto_clave = 'enfermero'),
  pr_top(country_puesto[puesto_secretario == 1], 'tipo_cargo_clean', 50) %>% mutate(puesto_clave = 'secretario')
)) %>% 
  select(-c(n, percentage))


write_flex(temp,
           file.path('data', 'clean', 'aggregates', 'country classified positions.xlsx'))

beep()



### + puesto_clave  --------------------------------------------------------------------------------------------
# NOTE: Create a single column with key position - note some puesto dummy columns overlap, so we need some more cleaning

### solve conflicts that persists like 'informatico biomedico' or 'profesor informatico'
# NOTE: add position_id if this column is added
country_puesto = country_puesto %>% mutate(
  across(c(puesto_enfermero, puesto_medico, puesto_docente), ~ifelse(. == 1 & puesto_secretario == 1, 0, .)), #  if both secretary and something else -> secretary
  across(c(puesto_enfermero, puesto_medico), ~ifelse(. == 1 & puesto_docente == 1, 0, .)), #  if both teacher and something else -> teacher
  across(c(puesto_medico), ~ifelse(. == 1 & puesto_enfermero == 1, 0, .)) #  if both enfermero and something else -> enfermero
  # across(c(puesto_medico), ~ifelse(. == 1 & puesto_it == 1, 0, .)), # if both medico and IT -> IT
)


### checks -> all key positions coded exclusively?
# country_puesto = country_puesto %>% mutate(n = rowSums(across(starts_with('puesto_'))))
# sf(country_puesto$n) # should be just 0's and 1's
# country_puesto %>% filter(n>1) %>% View # if not - see where is the problem

### find max. number of key positions per person
key_max = country_puesto %>% select(c(puesto_docente, puesto_medico, puesto_enfermero, puesto_secretario)) %>%
  mutate(n = rowSums(across(starts_with('puesto_')))) %>% pull(n) %>% max_miss()

pr_top(country_puesto[puesto_enfermero==1], 'tipo_cargo_clean', 30)

### if just 1, then assign key positions as labels to a new column and remove the binary 0-1 ones
if(key_max == 1){
  
  country_puesto = country_puesto %>% 
    mutate(
      puesto_clave = dplyr::case_when(
        puesto_docente == 1 ~ "docente",
        # puesto_it == 1 ~ "IT profesional",
        puesto_medico == 1 ~ "medico",
        puesto_enfermero == 1 & !grepl('\\btecn\\b|\\btecnico\\b|\\batencion\\b|auxiliar', tipo_cargo_clean) ~ "enfermero",
        # puesto_enfermero == 1 & grepl('\\bauxiliar\\b', tipo_cargo_clean) ~ "enfermero auxiliar", #\\b necessary!
        puesto_enfermero == 1 & grepl('\\btecn\\b|\\btecnico\\b|\\batencion\\b|auxiliar', tipo_cargo_clean) ~ "tecnico de enfermero (incl. atencion clinico)",
        puesto_secretario == 1 ~ "secretario",
        .default = NA_character_  
      )) %>% 
    dplyr::select(-starts_with(
      'puesto'), tipo_cargo_clean, puesto_clave)
  
}else{ ### ... else print that there is an issue and break the code
  stop('(custom message) Error: overlapping key positions')
}

pr_na(country_puesto$puesto_clave)
pr_top(country_puesto[str_detect(puesto_clave, '^enfermero'),], 'tipo_cargo_clean', 30)
pr_top(country_puesto[str_detect(puesto_clave, '^tecnico de enf'),], 'tipo_cargo_clean', 30)


### assign proper label to NAs
if(!any(grepl('data.table', class(country_puesto)))){setDT(country_puesto)}
country_puesto[, puesto_clave := fifelse(is.na(puesto_clave), 'otro tipo de cargo', puesto_clave)]




# '  -------------------------------------------------------------------------------------------------------------------------------------------------------
# RE-COMBINE WITH FULL -------------------------------------------------------------------------------------------------------------------------------------------------------------
# NOTE: grado values in the relevant column itself is somewhat too bit aggregated, so use the CLEANED puesto column to adjust

# leave only columns to be added
country_puesto = country_puesto[, .(row_id_org = unlist(id_nest)), by = setdiff(names(country_puesto), "id_nest")]
country_puesto = unique(country_puesto[, .(row_id_org, tipo_cargo_clean, puesto_clave, tipo_estamento_comprimido)])

nrow(country_puesto) == fdistinct(country_puesto$row_id_org)


# read the full data (only needed columns)
col_names = names(open_dataset(file.path(main_dir, 'data', 'intermediate', "country_04_limpiar_cols.parquet")))[]
col_names = col_names[!col_names %in% c('tipo_estamento_comprimido', 'tipo_cargo_clean', 'puesto_clave')]

t0=Sys.time()
country = read_flex(file.path(main_dir, 'data', 'intermediate', "country_04_limpiar_cols"), format = 'parquet',
                  col_select = col_names)
t1=Sys.time()
t1-t0 # ~43 seconds?

### combine
if(all(country$row_id_org %in% country_puesto$row_id_org)){ # all row ID from 'country' need to be in 'country_puesto'
  
  setkey(country_puesto, row_id_org)  
  setkey(country, row_id_org)  
  gc()
  
  country = country_puesto[country, on = 'row_id_org']
  
}else{
  stop('Not all row IDs found!')
}



### (*) checks - rank  ----------------------------------------------------------------------------------------------------------------------------------------------



# checks > gender pay gap (should be ~1.2)
country[, fmean(pago_bruto[genero == 'hombre'])/fmean(pago_bruto[genero == 'mujer']), by = .(anyo)]

# checks > gender directivo gap (women should be ~40-44% of directivos)
country[,uniqueN(name_full_original[tipo_estamento_comprimido == "directivo" &  genero == 'mujer'])/
        uniqueN(name_full_original[tipo_estamento_comprimido == "directivo"]),
      by = .(anyo)]







### (*) checks - puesto  ----------------------------------------------------------------------------------------------------------------------------------------------
### distribution of key positions
# country[, .(n=uniqueN(name_full_original)), by = .(puesto_clave, anyo)] %>% arrange(puesto_clave, anyo)
# country_final[, .(n=uniqueN(person_id)), by = .(puesto_clave, anyo)] %>% arrange(puesto_clave, anyo)

pr_na(country$puesto_clave)
pr_isna(country$tipo_cargo_clean)
pr_isna(country$tipo_cargo)

pr_top(country[puesto_clave=='docente'], 'tipo_cargo_clean', 30)
# pr_top(country[is.na(tipo_cargo) & is.na(tipo_calificacion)], 'tipo_cargo_clean', 30)


### (*) plot position timeline --------------------------------------------------------------------------------------------------------------------------------------------------------


### how many IDs/names per key position by year?
# should be, very broadly
# teachers = 146k; nurses = 27k (+ 16k auxillary); doctors = 40k; 
# country_final[,.(n=uniqueN(person_id)), by=.(anyo, puesto_clave)] %>% arrange(puesto_clave,anyo)
country[, .(n=uniqueN(name_full_original)), by=.(anyo, puesto_clave)] %>% arrange(puesto_clave,anyo)


### checks > does the distribution of key positions by sector look right?
# (e.g. no teachers outside education/municipal sectors, no doctors outside health sectors)
tapply(country$puesto_clave, country$organismo_sector_comprimido, pr_na)
temp = country[, fdistinct(id), by=.(puesto_clave, anyo)] 
temp %>% arrange(puesto_clave, anyo)


### summarise for the plot - key positions by year
dta_plot = country[, .(n=uniqueN(name_full)), by = .(anyo, dataset, puesto_clave)] %>% arrange(puesto_clave, anyo)

### plot
g1 = ggplot(dta_plot %>%  mutate(puesto_clave = str_to_title(puesto_clave)),
            aes(x=anyo, y = n, 
                color = puesto_clave, group = paste(puesto_clave, dataset)))+
  geom_hline(aes(yintercept = 0)) + 
  
  geom_line(linewidth = 1.6) +
  geom_point(aes(shape = dataset), size = 4.4, fill = 'white', stroke = 1.6) +
  # geom_text(aes(label = paste0(round(n/1e3,0), 'k')),
  #           size = 3.8, color = 'black') +
  
  
  facet_wrap(~str_wrap_br(puesto_clave, 17), scales = 'free') +
  
  scale_y_continuous(labels = comma,
                     limits = c(0,NA),
                     expand = expansion(mult = c(0.04,.2))) +
  scale_x_continuous( expand = expansion(mult = c(0.1,.1)), breaks = seq(2019,2024,1)) +
  scale_color_manual(name = '',
                     values = palette2) +
  scale_shape_manual(name = 'Tipo de contrato  ', 
                     values = c(21,22,23,24)) +
  
  guides(color = 'none', shape = guide_legend(title.position = 'top', override.aes = list(color = rep('black', 4)))) +

  
  labs(
    title = gsub('\n', '<br>',
                 str_wrap_br(paste0('<b>Figure XX. Employment in Key Positions in Country',
                                    '</b><br>'), 85)),
    x = '<br><b>Year</b>',
    y = '<br><b>Number of employees</b><br>',
    
    caption = paste0("<br><b>Notes:</b> Values calculated as total number of unique 
      employees working in each key position in a given year<br>"))+
  theme(
    axis.line.y = element_line(),
    axis.line.x = element_blank(),
    axis.text.x = element_markdown(size = 13),
    panel.border = element_blank(),
    
    legend.title = element_markdown(size = 25, face = 'bold'),
    legend.text = element_markdown(size = 25),
    legend.key.spacing.x = unit(0.7, 'cm'),
    legend.position = 'top'
    )

g1

ggsave(g1, 
       filename = file.path(main_dir, 'Figures', 'Overview - employment in Key Positions in Country.png'),
       width = 40, height = 25, unit = 'cm',
       scale = 1) 


# pr_top(country_final[puesto_clave == 'enfermero'], 'puesto_nombre', 30)
# pr_top(country_puesto[puesto_enfermero ==1], 'tipo_cargo', 30)

# '  ---------------------------------------------------------------------------------------------------------------------------------------------------------

# > SAVE -----------------------------------------------------------------------------------------------------------------------------------
#

# save
write_flex(country, file.path(main_dir, 'data', 'intermediate', 'country_05_limpiar_puesto'), format=format1)

exec_time_fun('exec_time')





#
# FIN DEL CÓDIGO  --------------------------------------------------------------------------------------------
# 